<?php
/**
 * Unit Tests for ComprehendService
 */

use Redact\Classes\ComprehendService;

class ComprehendServiceTest extends BaseTest
{
    private array $awsCredentials;
    
    public function runTests(): void
    {
        $this->awsCredentials = getConfig('AWS_Credentials');
        
        if (!$this->awsCredentials || !isset($this->awsCredentials['key'])) {
            echo "  ⏭️  Skipping Comprehend tests (AWS credentials not configured)\n";
            return;
        }
        
        $this->testServiceInitialization();
        $this->testPIIDetection();
        $this->testNameDetection();
        $this->testBankDetailsDetection();
        $this->testMultiplePIITypes();
    }
    
    private function testServiceInitialization(): void
    {
        $comprehend = new ComprehendService(
            $this->awsCredentials['key'],
            $this->awsCredentials['secret'],
            'us-east-1'
        );
        
        $this->runner->assertNotNull($comprehend, 'ComprehendService should initialize');
    }
    
    private function testPIIDetection(): void
    {
        $comprehend = new ComprehendService(
            $this->awsCredentials['key'],
            $this->awsCredentials['secret']
        );
        
        $text = "My name is John Smith and my email is john@example.com";
        $result = $comprehend->detectPiiEntities($text);
        
        $this->runner->assertTrue($result['success'], 'PII detection should succeed');
        $this->runner->assertNotNull($result['data'] ?? null, 'Should return data');
        $this->runner->assertNotNull($result['data']['Entities'] ?? null, 'Should return entities');
    }
    
    private function testNameDetection(): void
    {
        $comprehend = new ComprehendService(
            $this->awsCredentials['key'],
            $this->awsCredentials['secret']
        );
        
        $text = "Seyhan Beytekin";
        $result = $comprehend->detectPiiEntities($text);
        
        if ($result['success']) {
            $entities = $result['data']['Entities'] ?? [];
            
            $hasName = false;
            foreach ($entities as $entity) {
                if ($entity['Type'] === 'NAME') {
                    $hasName = true;
                    break;
                }
            }
            
            $this->runner->assertTrue($hasName, 'Should detect NAME entities');
        }
    }
    
    private function testBankDetailsDetection(): void
    {
        $comprehend = new ComprehendService(
            $this->awsCredentials['key'],
            $this->awsCredentials['secret']
        );
        
        $text = "Sort Code: 30-99-08  Account Number: 10253568";
        $result = $comprehend->detectPiiEntities($text);
        
        if ($result['success']) {
            $entities = $result['data']['Entities'] ?? [];
            
            $this->runner->assertGreaterThan(0, count($entities), 'Should detect entities in bank details with context');
            
            // Check for BANK_ACCOUNT_NUMBER or similar
            $hasBankInfo = false;
            foreach ($entities as $entity) {
                if (in_array($entity['Type'], ['BANK_ACCOUNT_NUMBER', 'BANK_ROUTING', 'PIN'])) {
                    $hasBankInfo = true;
                    break;
                }
            }
            
            // Note: Bank details may or may not be detected depending on AWS Comprehend's confidence
            if ($hasBankInfo) {
                $this->runner->assertTrue(true, 'Bank account information detected');
            } else {
                echo "  ℹ️  Bank details not detected (may vary by context)\n";
            }
        }
    }
    
    private function testMultiplePIITypes(): void
    {
        $comprehend = new ComprehendService(
            $this->awsCredentials['key'],
            $this->awsCredentials['secret']
        );
        
        $text = "Contact John Smith at john@example.com or call 555-123-4567. Address: 123 Main Street, London.";
        $result = $comprehend->detectPiiEntities($text);
        
        if ($result['success']) {
            $entities = $result['data']['Entities'] ?? [];
            
            $types = array_unique(array_column($entities, 'Type'));
            
            $this->runner->assertGreaterThan(1, count($types), 'Should detect multiple PII types');
            
            // Check for common types
            $hasName = in_array('NAME', $types);
            $hasEmail = in_array('EMAIL', $types);
            $hasPhone = in_array('PHONE', $types);
            $hasAddress = in_array('ADDRESS', $types);
            
            if ($hasName) $this->runner->assertTrue(true, 'Detected NAME');
            if ($hasEmail) $this->runner->assertTrue(true, 'Detected EMAIL');
            if ($hasPhone) $this->runner->assertTrue(true, 'Detected PHONE');
            if ($hasAddress) $this->runner->assertTrue(true, 'Detected ADDRESS');
        }
    }
}

