<?php
/**
 * Thread Manager
 * Manages user threads/sessions for segregated data storage
 * Prevents cross-contamination of PII data between users
 */

namespace Redact\Classes;

class ThreadManager
{
    private string $dataDir;
    private string $threadsIndexFile;
    private int $threadExpiryDays = 30;
    
    public function __construct(string $dataDir = null, int $threadExpiryDays = 30)
    {
        $this->dataDir = $dataDir ?? __DIR__ . '/../../data';
        $this->threadsIndexFile = $this->dataDir . '/threads_index.json';
        $this->threadExpiryDays = $threadExpiryDays;
        
        // Ensure data directory exists
        if (!is_dir($this->dataDir)) {
            mkdir($this->dataDir, 0755, true);
        }
        
        // Auto-cleanup old threads on initialization
        $this->cleanupExpiredThreads();
    }
    
    /**
     * Create a new thread
     *
     * @param array $metadata Optional metadata (user info, description, etc.)
     * @return array Thread information including private key
     */
    public function createThread(array $metadata = []): array
    {
        $threadId = $this->generateThreadId();
        $privateKey = $this->generatePrivateKey();
        $threadDir = $this->getThreadDirectory($threadId);
        
        // Create thread directory
        if (!is_dir($threadDir)) {
            mkdir($threadDir, 0755, true);
        }
        
        $thread = [
            'thread_id' => $threadId,
            'private_key_hash' => password_hash($privateKey, PASSWORD_BCRYPT),
            'created_at' => date('Y-m-d H:i:s'),
            'last_activity' => date('Y-m-d H:i:s'),
            'document_count' => 0,
            'total_pii_found' => 0,
            'total_api_calls' => 0,
            'metadata' => $metadata
        ];
        
        // Save thread info
        $this->saveThreadInfo($threadId, $thread);
        
        // Update threads index (without private key hash for security)
        $indexThread = $thread;
        unset($indexThread['private_key_hash']);
        $this->addToIndex($indexThread);
        
        return [
            'success' => true,
            'thread_id' => $threadId,
            'private_key' => $privateKey, // Only returned once!
            'thread' => $indexThread
        ];
    }
    
    /**
     * Get thread information
     *
     * @param string $threadId Thread ID
     * @return array|null Thread data or null if not found
     */
    public function getThread(string $threadId): ?array
    {
        if (!$this->threadExists($threadId)) {
            return null;
        }
        
        $threadFile = $this->getThreadDirectory($threadId) . '/thread_info.json';
        
        if (!file_exists($threadFile)) {
            return null;
        }
        
        $data = file_get_contents($threadFile);
        return json_decode($data, true);
    }
    
    /**
     * Update thread activity timestamp
     *
     * @param string $threadId Thread ID
     * @return bool Success status
     */
    public function updateThreadActivity(string $threadId): bool
    {
        $thread = $this->getThread($threadId);
        
        if (!$thread) {
            return false;
        }
        
        $thread['last_activity'] = date('Y-m-d H:i:s');
        
        $this->saveThreadInfo($threadId, $thread);
        $this->updateIndex($thread);
        
        return true;
    }
    
    /**
     * Increment thread statistics
     *
     * @param string $threadId Thread ID
     * @param array $stats Statistics to increment (document_count, total_pii_found, total_api_calls)
     * @return bool Success status
     */
    public function incrementThreadStats(string $threadId, array $stats): bool
    {
        $thread = $this->getThread($threadId);
        
        if (!$thread) {
            return false;
        }
        
        if (isset($stats['document_count'])) {
            $thread['document_count'] += $stats['document_count'];
        }
        
        if (isset($stats['total_pii_found'])) {
            $thread['total_pii_found'] += $stats['total_pii_found'];
        }
        
        if (isset($stats['total_api_calls'])) {
            $thread['total_api_calls'] += $stats['total_api_calls'];
        }
        
        $thread['last_activity'] = date('Y-m-d H:i:s');
        
        $this->saveThreadInfo($threadId, $thread);
        $this->updateIndex($thread);
        
        return true;
    }
    
    /**
     * List all threads
     *
     * @param bool $includeExpired Include expired threads in list
     * @return array Array of threads
     */
    public function listThreads(bool $includeExpired = false): array
    {
        $index = $this->loadIndex();
        
        if (!$includeExpired) {
            $index['threads'] = array_filter($index['threads'], function($thread) {
                return !$this->isThreadExpired($thread);
            });
        }
        
        // Sort by last activity (newest first)
        usort($index['threads'], function($a, $b) {
            return strtotime($b['last_activity']) - strtotime($a['last_activity']);
        });
        
        return $index['threads'];
    }
    
    /**
     * Delete a thread and all its data
     *
     * @param string $threadId Thread ID
     * @param string|null $privateKey Private key for authentication (null skips auth check)
     * @return array Result with success status
     */
    public function deleteThread(string $threadId, ?string $privateKey = null): array
    {
        if (!$this->threadExists($threadId)) {
            return ['success' => false, 'error' => 'Thread not found'];
        }
        
        // Verify private key if provided
        if ($privateKey !== null && !$this->verifyPrivateKey($threadId, $privateKey)) {
            return ['success' => false, 'error' => 'Invalid private key'];
        }
        
        $threadDir = $this->getThreadDirectory($threadId);
        
        // Delete all files in thread directory
        $this->deleteDirectory($threadDir);
        
        // Remove from index
        $this->removeFromIndex($threadId);
        
        return [
            'success' => true,
            'thread_id' => $threadId,
            'message' => 'Thread and all associated data deleted'
        ];
    }
    
    /**
     * Check if thread exists
     *
     * @param string $threadId Thread ID
     * @return bool True if exists
     */
    public function threadExists(string $threadId): bool
    {
        $threadDir = $this->getThreadDirectory($threadId);
        return is_dir($threadDir);
    }
    
    /**
     * Check if thread is valid (exists and not expired)
     *
     * @param string $threadId Thread ID
     * @return bool True if valid
     */
    public function isThreadValid(string $threadId): bool
    {
        if (!$this->threadExists($threadId)) {
            return false;
        }
        
        $thread = $this->getThread($threadId);
        
        if (!$thread) {
            return false;
        }
        
        return !$this->isThreadExpired($thread);
    }
    
    /**
     * Get thread directory path
     *
     * @param string $threadId Thread ID
     * @return string Directory path
     */
    public function getThreadDirectory(string $threadId): string
    {
        return $this->dataDir . '/' . $threadId;
    }
    
    /**
     * Get thread cache directory path
     *
     * @param string $threadId Thread ID
     * @return string Cache directory path
     */
    public function getThreadCacheDirectory(string $threadId): string
    {
        $cacheDir = $this->getThreadDirectory($threadId) . '/cache';
        
        if (!is_dir($cacheDir)) {
            mkdir($cacheDir, 0755, true);
        }
        
        return $cacheDir;
    }
    
    /**
     * Cleanup expired threads (older than configured days)
     *
     * @return array Cleanup results
     */
    public function cleanupExpiredThreads(): array
    {
        $threads = $this->listThreads(true); // Include expired
        $deletedCount = 0;
        $deletedThreads = [];
        
        foreach ($threads as $thread) {
            if ($this->isThreadExpired($thread)) {
                $result = $this->deleteThread($thread['thread_id']);
                if ($result['success']) {
                    $deletedCount++;
                    $deletedThreads[] = $thread['thread_id'];
                }
            }
        }
        
        return [
            'success' => true,
            'deleted_count' => $deletedCount,
            'deleted_threads' => $deletedThreads
        ];
    }
    
    /**
     * Get statistics about all threads
     *
     * @return array Statistics
     */
    public function getStatistics(): array
    {
        $threads = $this->listThreads(false);
        
        $stats = [
            'total_threads' => count($threads),
            'total_documents' => 0,
            'total_pii_found' => 0,
            'total_api_calls' => 0,
            'oldest_thread' => null,
            'newest_thread' => null,
            'most_active_thread' => null
        ];
        
        if (empty($threads)) {
            return $stats;
        }
        
        foreach ($threads as $thread) {
            $stats['total_documents'] += $thread['document_count'] ?? 0;
            $stats['total_pii_found'] += $thread['total_pii_found'] ?? 0;
            $stats['total_api_calls'] += $thread['total_api_calls'] ?? 0;
        }
        
        // Find oldest and newest
        $stats['oldest_thread'] = $threads[count($threads) - 1]; // Sorted by last_activity desc
        $stats['newest_thread'] = $threads[0];
        
        // Find most active (by document count)
        $mostActive = $threads[0];
        foreach ($threads as $thread) {
            if (($thread['document_count'] ?? 0) > ($mostActive['document_count'] ?? 0)) {
                $mostActive = $thread;
            }
        }
        $stats['most_active_thread'] = $mostActive;
        
        return $stats;
    }
    
    // ========== Private Helper Methods ==========
    
    /**
     * Generate unique thread ID
     *
     * @return string Thread ID
     */
    private function generateThreadId(): string
    {
        return 'thread_' . bin2hex(random_bytes(16));
    }
    
    /**
     * Generate secure private key
     *
     * @return string Private key
     */
    private function generatePrivateKey(): string
    {
        return 'key_' . bin2hex(random_bytes(32));
    }
    
    /**
     * Verify private key for thread
     *
     * @param string $threadId Thread ID
     * @param string $privateKey Private key to verify
     * @return bool True if valid
     */
    public function verifyPrivateKey(string $threadId, string $privateKey): bool
    {
        $thread = $this->getThread($threadId);
        
        if (!$thread || !isset($thread['private_key_hash'])) {
            return false;
        }
        
        return password_verify($privateKey, $thread['private_key_hash']);
    }
    
    /**
     * Check if thread is expired
     *
     * @param array $thread Thread data
     * @return bool True if expired
     */
    private function isThreadExpired(array $thread): bool
    {
        $lastActivity = strtotime($thread['last_activity']);
        $expiryTime = time() - ($this->threadExpiryDays * 24 * 60 * 60);
        
        return $lastActivity < $expiryTime;
    }
    
    /**
     * Save thread info to file
     *
     * @param string $threadId Thread ID
     * @param array $thread Thread data
     */
    private function saveThreadInfo(string $threadId, array $thread): void
    {
        $threadFile = $this->getThreadDirectory($threadId) . '/thread_info.json';
        file_put_contents($threadFile, json_encode($thread, JSON_PRETTY_PRINT));
    }
    
    /**
     * Load threads index
     *
     * @return array Index data
     */
    private function loadIndex(): array
    {
        if (!file_exists($this->threadsIndexFile)) {
            return ['threads' => [], 'last_updated' => date('Y-m-d H:i:s')];
        }
        
        $data = file_get_contents($this->threadsIndexFile);
        return json_decode($data, true) ?? ['threads' => [], 'last_updated' => date('Y-m-d H:i:s')];
    }
    
    /**
     * Save threads index
     *
     * @param array $index Index data
     */
    private function saveIndex(array $index): void
    {
        $index['last_updated'] = date('Y-m-d H:i:s');
        file_put_contents($this->threadsIndexFile, json_encode($index, JSON_PRETTY_PRINT));
    }
    
    /**
     * Add thread to index
     *
     * @param array $thread Thread data
     */
    private function addToIndex(array $thread): void
    {
        $index = $this->loadIndex();
        $index['threads'][] = $thread;
        $this->saveIndex($index);
    }
    
    /**
     * Update thread in index
     *
     * @param array $thread Thread data
     */
    private function updateIndex(array $thread): void
    {
        $index = $this->loadIndex();
        
        foreach ($index['threads'] as $i => $t) {
            if ($t['thread_id'] === $thread['thread_id']) {
                $index['threads'][$i] = $thread;
                break;
            }
        }
        
        $this->saveIndex($index);
    }
    
    /**
     * Remove thread from index
     *
     * @param string $threadId Thread ID
     */
    private function removeFromIndex(string $threadId): void
    {
        $index = $this->loadIndex();
        
        $index['threads'] = array_filter($index['threads'], function($thread) use ($threadId) {
            return $thread['thread_id'] !== $threadId;
        });
        
        $index['threads'] = array_values($index['threads']); // Re-index array
        
        $this->saveIndex($index);
    }
    
    /**
     * Recursively delete directory
     *
     * @param string $dir Directory path
     */
    private function deleteDirectory(string $dir): void
    {
        if (!is_dir($dir)) {
            return;
        }
        
        $files = array_diff(scandir($dir), ['.', '..']);
        
        foreach ($files as $file) {
            $path = $dir . '/' . $file;
            
            if (is_dir($path)) {
                $this->deleteDirectory($path);
            } else {
                unlink($path);
            }
        }
        
        rmdir($dir);
    }
}

