<?php
/**
 * AWS Comprehend Service
 * Handles all AWS Comprehend API interactions for PII detection
 */

namespace Redact\Classes;

class ComprehendService
{
    private string $awsKey;
    private string $awsSecret;
    private string $region;
    
    public function __construct(string $awsKey, string $awsSecret, string $region = 'us-east-1')
    {
        $this->awsKey = $awsKey;
        $this->awsSecret = $awsSecret;
        $this->region = $region;
    }
    
    /**
     * Detect PII entities in text
     *
     * @param string $text Text to analyze
     * @param string $languageCode Language code (default: 'en')
     * @return array Result array with success status and entities
     */
    public function detectPiiEntities(string $text, string $languageCode = 'en'): array
    {
        $endpoint = "https://comprehend.{$this->region}.amazonaws.com/";
        $service = 'comprehend';
        
        $payload = json_encode([
            'Text' => $text,
            'LanguageCode' => $languageCode
        ]);
        
        $headers = $this->createAWSSignature(
            'POST',
            $endpoint,
            $service,
            $payload,
            'Comprehend_20171127.DetectPiiEntities'
        );
        
        return $this->executeCurlRequest($endpoint, $headers, $payload);
    }
    
    /**
     * Create AWS Signature Version 4
     *
     * @param string $method HTTP method
     * @param string $endpoint API endpoint
     * @param string $service AWS service name
     * @param string $payload Request payload
     * @param string|null $target API target action
     * @return array Headers array
     */
    private function createAWSSignature(
        string $method,
        string $endpoint,
        string $service,
        string $payload,
        ?string $target = null
    ): array {
        $algorithm = 'AWS4-HMAC-SHA256';
        $amzDate = gmdate('Ymd\THis\Z');
        $dateStamp = gmdate('Ymd');
        
        $urlParts = parse_url($endpoint);
        $host = $urlParts['host'];
        $uri = $urlParts['path'] ?? '/';
        
        $canonicalHeaders = "host:$host\n" . "x-amz-date:$amzDate\n";
        $signedHeaders = 'host;x-amz-date';
        
        $payloadHash = hash('sha256', $payload);
        $canonicalRequest = "$method\n$uri\n\n$canonicalHeaders\n$signedHeaders\n$payloadHash";
        
        $credentialScope = "$dateStamp/{$this->region}/$service/aws4_request";
        $stringToSign = "$algorithm\n$amzDate\n$credentialScope\n" . hash('sha256', $canonicalRequest);
        
        $kDate = hash_hmac('sha256', $dateStamp, "AWS4{$this->awsSecret}", true);
        $kRegion = hash_hmac('sha256', $this->region, $kDate, true);
        $kService = hash_hmac('sha256', $service, $kRegion, true);
        $kSigning = hash_hmac('sha256', 'aws4_request', $kService, true);
        
        $signature = hash_hmac('sha256', $stringToSign, $kSigning);
        
        $authorizationHeader = "$algorithm Credential={$this->awsKey}/$credentialScope, SignedHeaders=$signedHeaders, Signature=$signature";
        
        $headers = [
            'Authorization' => $authorizationHeader,
            'x-amz-date' => $amzDate,
            'Content-Type' => 'application/x-amz-json-1.1'
        ];
        
        if ($target) {
            $headers['x-amz-target'] = $target;
        }
        
        return $headers;
    }
    
    /**
     * Execute cURL request
     *
     * @param string $endpoint API endpoint
     * @param array $headers Request headers
     * @param string $payload Request payload
     * @return array Result with success status and data
     */
    private function executeCurlRequest(string $endpoint, array $headers, string $payload): array
    {
        $ch = curl_init($endpoint);
        
        $curlHeaders = [];
        foreach ($headers as $key => $value) {
            $curlHeaders[] = "$key: $value";
        }
        
        curl_setopt($ch, CURLOPT_HTTPHEADER, $curlHeaders);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $payload);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 60);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curlError = curl_error($ch);
        curl_close($ch);
        
        if ($httpCode == 200) {
            return [
                'success' => true,
                'data' => json_decode($response, true)
            ];
        } else {
            return [
                'success' => false,
                'error' => $response,
                'curl_error' => $curlError,
                'http_code' => $httpCode
            ];
        }
    }
}

