<?php
/**
 * API Configuration
 */

// Enable error reporting for development
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);
ini_set('error_log', __DIR__ . '/api_errors.log');

// CORS Configuration
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *'); // Change to specific domain in production
header('Access-Control-Allow-Methods: POST, GET, OPTIONS, DELETE');
header('Access-Control-Allow-Headers: Content-Type, Authorization, X-Thread-ID, X-Private-Key');
header('Access-Control-Max-Age: 3600');

// Handle preflight OPTIONS request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// Load classes
require_once __DIR__ . '/../src/classes/autoload.php';
require_once __DIR__ . '/../src/config/config.php';

// API Configuration
define('API_VERSION', '1.0');
define('API_MAX_FILE_SIZE', 10 * 1024 * 1024); // 10MB
define('API_RATE_LIMIT', 100); // requests per hour per thread

/**
 * Send JSON response
 */
function sendResponse($data, $statusCode = 200) {
    http_response_code($statusCode);
    echo json_encode($data, JSON_PRETTY_PRINT);
    exit;
}

/**
 * Send error response
 */
function sendError($message, $statusCode = 400, $details = null) {
    $response = [
        'success' => false,
        'error' => $message
    ];
    
    if ($details !== null) {
        $response['details'] = $details;
    }
    
    sendResponse($response, $statusCode);
}

/**
 * Get request body as JSON
 */
function getRequestBody() {
    $body = file_get_contents('php://input');
    $data = json_decode($body, true);
    
    if (json_last_error() !== JSON_ERROR_NONE) {
        sendError('Invalid JSON in request body', 400);
    }
    
    return $data ?? [];
}

/**
 * Validate thread authentication
 */
function validateThreadAuth($threadId, $privateKey) {
    if (empty($threadId) || empty($privateKey)) {
        sendError('thread_id and private_key are required', 401);
    }
    
    // Load thread authentication
    $authFile = __DIR__ . '/../data/' . $threadId . '/auth.json';
    
    if (!file_exists($authFile)) {
        sendError('Invalid thread_id or private_key', 401);
    }
    
    $authData = json_decode(file_get_contents($authFile), true);
    
    if (!$authData || !isset($authData['private_key'])) {
        sendError('Thread authentication error', 500);
    }
    
    // Verify private key with timing-safe comparison
    if (!hash_equals($authData['private_key'], $privateKey)) {
        sendError('Invalid thread_id or private_key', 401);
    }
    
    return true;
}

/**
 * Log API request
 */
function logRequest($endpoint, $threadId = null) {
    $logEntry = [
        'timestamp' => date('Y-m-d H:i:s'),
        'endpoint' => $endpoint,
        'method' => $_SERVER['REQUEST_METHOD'],
        'ip' => $_SERVER['REMOTE_ADDR'] ?? 'unknown',
        'thread_id' => $threadId,
        'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'unknown'
    ];
    
    $logFile = __DIR__ . '/api_access.log';
    file_put_contents($logFile, json_encode($logEntry) . "\n", FILE_APPEND);
}

