<?php
/**
 * Test Thread Management System
 * Demonstrates thread creation, caching, and data segregation
 */

require_once __DIR__ . '/../autoload.php';

use Redact\Classes\ThreadManager;
use Redact\Classes\RegistryManager;

echo "\n╔════════════════════════════════════════════════════════════════╗\n";
echo "║              THREAD MANAGEMENT SYSTEM TEST                     ║\n";
echo "╚════════════════════════════════════════════════════════════════╝\n\n";

$dataDir = __DIR__ . '/../../../data';
$threadManager = new ThreadManager($dataDir, 30);

// Test 1: Create Threads
echo "📋 TEST 1: Creating Threads\n";
echo str_repeat("-", 70) . "\n";

$thread1 = $threadManager->createThread(['user_id' => 'user_001', 'name' => 'Legal Documents']);
$thread2 = $threadManager->createThread(['user_id' => 'user_002', 'name' => 'HR Documents']);

echo "✅ Thread 1 created: {$thread1['thread_id']}\n";
echo "✅ Thread 2 created: {$thread2['thread_id']}\n\n";

// Test 2: List Threads
echo "📋 TEST 2: Listing Threads\n";
echo str_repeat("-", 70) . "\n";

$threads = $threadManager->listThreads();
echo "Total threads: " . count($threads) . "\n";

foreach ($threads as $thread) {
    echo "  - {$thread['thread_id']}\n";
    echo "    Created: {$thread['created_at']}\n";
    echo "    Documents: {$thread['document_count']}\n";
}
echo "\n";

// Test 3: Thread Segregation (Simulate PII caching)
echo "📋 TEST 3: Data Segregation & Caching\n";
echo str_repeat("-", 70) . "\n";

$threadId1 = $thread1['thread_id'];
$threadId2 = $thread2['thread_id'];

// Create registry for thread 1
$cacheDir1 = $threadManager->getThreadCacheDirectory($threadId1);
$registry1 = new RegistryManager($threadId1, $cacheDir1);

// Simulate finding PII in thread 1
$mockPII1 = [
    'John' => [['type' => 'NAME', 'score' => 0.99, 'context' => 'John Smith']],
    'Smith' => [['type' => 'NAME', 'score' => 0.99, 'context' => 'John Smith']]
];

// Manually set processed words (simulating PII detection)
$reflection = new ReflectionClass($registry1);
$property = $reflection->getProperty('processedWordBlocks');
$property->setAccessible(true);
$property->setValue($registry1, $mockPII1);

// Save cache for thread 1
$registry1->saveCachedRegistry();
echo "✅ Thread 1: Saved cache with 2 PII words (John, Smith)\n";

// Create registry for thread 2
$cacheDir2 = $threadManager->getThreadCacheDirectory($threadId2);
$registry2 = new RegistryManager($threadId2, $cacheDir2);

// Simulate finding DIFFERENT PII in thread 2
$mockPII2 = [
    'Jane' => [['type' => 'NAME', 'score' => 0.99, 'context' => 'Jane Doe']],
    'Doe' => [['type' => 'NAME', 'score' => 0.99, 'context' => 'Jane Doe']]
];

$property->setValue($registry2, $mockPII2);
$registry2->saveCachedRegistry();
echo "✅ Thread 2: Saved cache with 2 PII words (Jane, Doe)\n\n";

// Test 4: Verify Segregation
echo "📋 TEST 4: Verify Data Segregation\n";
echo str_repeat("-", 70) . "\n";

// Reload registries
$registry1Reload = new RegistryManager($threadId1, $cacheDir1);
$registry2Reload = new RegistryManager($threadId2, $cacheDir2);

$stats1 = $registry1Reload->getCacheStatistics();
$stats2 = $registry2Reload->getCacheStatistics();

echo "Thread 1 cache:\n";
echo "  Cached words: {$stats1['cached_words']}\n";
echo "  Last updated: {$stats1['last_updated']}\n";

echo "Thread 2 cache:\n";
echo "  Cached words: {$stats2['cached_words']}\n";
echo "  Last updated: {$stats2['last_updated']}\n";

if ($stats1['cached_words'] === 2 && $stats2['cached_words'] === 2) {
    echo "✅ Data segregation verified! Each thread has isolated cache.\n\n";
} else {
    echo "❌ Data segregation failed!\n\n";
}

// Test 5: Update Thread Stats
echo "📋 TEST 5: Update Thread Statistics\n";
echo str_repeat("-", 70) . "\n";

$threadManager->incrementThreadStats($threadId1, [
    'document_count' => 1,
    'total_pii_found' => 25,
    'total_api_calls' => 10
]);

$threadManager->incrementThreadStats($threadId2, [
    'document_count' => 2,
    'total_pii_found' => 50,
    'total_api_calls' => 20
]);

$thread1Info = $threadManager->getThread($threadId1);
$thread2Info = $threadManager->getThread($threadId2);

echo "Thread 1 stats:\n";
echo "  Documents: {$thread1Info['document_count']}\n";
echo "  PII found: {$thread1Info['total_pii_found']}\n";
echo "  API calls: {$thread1Info['total_api_calls']}\n";

echo "Thread 2 stats:\n";
echo "  Documents: {$thread2Info['document_count']}\n";
echo "  PII found: {$thread2Info['total_pii_found']}\n";
echo "  API calls: {$thread2Info['total_api_calls']}\n\n";

// Test 6: Global Statistics
echo "📋 TEST 6: Global Statistics\n";
echo str_repeat("-", 70) . "\n";

$globalStats = $threadManager->getStatistics();

echo "Total threads: {$globalStats['total_threads']}\n";
echo "Total documents: {$globalStats['total_documents']}\n";
echo "Total PII found: {$globalStats['total_pii_found']}\n";
echo "Total API calls: {$globalStats['total_api_calls']}\n\n";

// Test 7: Thread Validation
echo "📋 TEST 7: Thread Validation\n";
echo str_repeat("-", 70) . "\n";

$isValid1 = $threadManager->isThreadValid($threadId1);
$isValid2 = $threadManager->isThreadValid($threadId2);
$isValidFake = $threadManager->isThreadValid('thread_fake123');

echo "Thread 1 valid: " . ($isValid1 ? '✅ Yes' : '❌ No') . "\n";
echo "Thread 2 valid: " . ($isValid2 ? '✅ Yes' : '❌ No') . "\n";
echo "Fake thread valid: " . ($isValidFake ? '✅ Yes' : '❌ No') . "\n\n";

// Test 8: Cleanup Test Threads
echo "📋 TEST 8: Cleanup\n";
echo str_repeat("-", 70) . "\n";

$delete1 = $threadManager->deleteThread($threadId1);
$delete2 = $threadManager->deleteThread($threadId2);

echo "Thread 1 deleted: " . ($delete1['success'] ? '✅ Yes' : '❌ No') . "\n";
echo "Thread 2 deleted: " . ($delete2['success'] ? '✅ Yes' : '❌ No') . "\n\n";

// Verify deletion
$threadsAfter = $threadManager->listThreads();
echo "Threads remaining: " . count($threadsAfter) . "\n\n";

echo "╔════════════════════════════════════════════════════════════════╗\n";
echo "║              ALL TESTS COMPLETED                                ║\n";
echo "╚════════════════════════════════════════════════════════════════╝\n\n";

echo "✅ Thread creation\n";
echo "✅ Thread listing\n";
echo "✅ Data segregation\n";
echo "✅ Persistent caching\n";
echo "✅ Statistics tracking\n";
echo "✅ Thread validation\n";
echo "✅ Thread deletion\n\n";

echo "🎉 Thread management system working perfectly!\n\n";

