<?php
/**
 * Unit Tests for TextractService
 */

use Redact\Classes\TextractService;
use Redact\Classes\FileHandler;

class TextractServiceTest extends BaseTest
{
    private string $testPdfPath;
    private array $awsCredentials;
    
    public function runTests(): void
    {
        $this->testPdfPath = __DIR__ . '/BeytekinS Payslips.pdf';
        $this->awsCredentials = getConfig('AWS_Credentials');
        
        if (!$this->awsCredentials || !isset($this->awsCredentials['key'])) {
            echo "  ⏭️  Skipping Textract tests (AWS credentials not configured)\n";
            return;
        }
        
        $this->testServiceInitialization();
        $this->testAnalyzeDocument();
        $this->testBlockExtraction();
        $this->testLayoutDetection();
    }
    
    private function testServiceInitialization(): void
    {
        $textract = new TextractService(
            $this->awsCredentials['key'],
            $this->awsCredentials['secret'],
            'us-east-1'
        );
        
        $this->runner->assertNotNull($textract, 'TextractService should initialize');
    }
    
    private function testAnalyzeDocument(): void
    {
        if (!extension_loaded('imagick')) {
            echo "  ⏭️  Skipping document analysis test (Imagick not installed)\n";
            return;
        }
        
        $textract = new TextractService(
            $this->awsCredentials['key'],
            $this->awsCredentials['secret']
        );
        
        $fileHandler = new FileHandler();
        $conversion = $fileHandler->convertToImages($this->testPdfPath, 'test.pdf');
        
        if (!$conversion['success'] || empty($conversion['images'])) {
            echo "  ⏭️  Skipping test (PDF conversion failed)\n";
            return;
        }
        
        $imageData = $conversion['images'][0];
        $result = $textract->analyzeDocument($imageData);
        
        $this->runner->assertTrue($result['success'], 'Textract analyze should succeed');
        $this->runner->assertNotNull($result['data'] ?? null, 'Should return data');
        $this->runner->assertNotNull($result['data']['Blocks'] ?? null, 'Should return blocks');
    }
    
    private function testBlockExtraction(): void
    {
        if (!extension_loaded('imagick')) {
            echo "  ⏭️  Skipping block extraction test (Imagick not installed)\n";
            return;
        }
        
        $textract = new TextractService(
            $this->awsCredentials['key'],
            $this->awsCredentials['secret']
        );
        
        $fileHandler = new FileHandler();
        $conversion = $fileHandler->convertToImages($this->testPdfPath, 'test.pdf');
        
        if (!$conversion['success'] || empty($conversion['images'])) {
            echo "  ⏭️  Skipping test (PDF conversion failed)\n";
            return;
        }
        
        $imageData = $conversion['images'][0];
        $result = $textract->analyzeDocument($imageData);
        
        if ($result['success']) {
            $blocks = $result['data']['Blocks'] ?? [];
            
            $this->runner->assertGreaterThan(0, count($blocks), 'Should extract blocks');
            
            // Check for different block types
            $hasWord = false;
            $hasLine = false;
            $hasLayout = false;
            
            foreach ($blocks as $block) {
                if (($block['BlockType'] ?? '') === 'WORD') $hasWord = true;
                if (($block['BlockType'] ?? '') === 'LINE') $hasLine = true;
                if (strpos($block['BlockType'] ?? '', 'LAYOUT_') === 0) $hasLayout = true;
            }
            
            $this->runner->assertTrue($hasWord, 'Should extract WORD blocks');
            $this->runner->assertTrue($hasLine, 'Should extract LINE blocks');
            $this->runner->assertTrue($hasLayout, 'Should extract LAYOUT blocks');
        }
    }
    
    private function testLayoutDetection(): void
    {
        if (!extension_loaded('imagick')) {
            echo "  ⏭️  Skipping layout detection test (Imagick not installed)\n";
            return;
        }
        
        $textract = new TextractService(
            $this->awsCredentials['key'],
            $this->awsCredentials['secret']
        );
        
        $fileHandler = new FileHandler();
        $conversion = $fileHandler->convertToImages($this->testPdfPath, 'test.pdf');
        
        if (!$conversion['success'] || empty($conversion['images'])) {
            echo "  ⏭️  Skipping test (PDF conversion failed)\n";
            return;
        }
        
        $imageData = $conversion['images'][0];
        $result = $textract->analyzeDocument($imageData);
        
        if ($result['success']) {
            $blocks = $result['data']['Blocks'] ?? [];
            
            $layoutTypes = [];
            foreach ($blocks as $block) {
                $type = $block['BlockType'] ?? '';
                if (strpos($type, 'LAYOUT_') === 0) {
                    $layoutTypes[] = $type;
                }
            }
            
            $this->runner->assertGreaterThan(0, count($layoutTypes), 'Should detect layout types');
        }
    }
}

