<?php
/**
 * API v1: Delete Thread
 * 
 * DELETE /api/v1/thread_delete.php
 * 
 * Body (JSON):
 * {
 *   "thread_id": "thread_abc123...",
 *   "private_key": "key_xyz789..."
 * }
 * 
 * Response:
 * {
 *   "success": true,
 *   "message": "Thread and all associated data deleted"
 * }
 */

require_once __DIR__ . '/../config.php';

use Redact\Classes\ThreadManager;

try {
    logRequest('thread_delete');
    
    // Allow POST or DELETE
    if (!in_array($_SERVER['REQUEST_METHOD'], ['POST', 'DELETE'])) {
        sendError('Method not allowed. Use POST or DELETE', 405);
    }
    
    // Get request body
    $body = getRequestBody();
    $threadId = $body['thread_id'] ?? null;
    $privateKey = $body['private_key'] ?? null;
    
    // Validate authentication
    validateThreadAuth($threadId, $privateKey);
    
    // Delete thread
    $threadManager = new ThreadManager();
    $result = $threadManager->deleteThread($threadId);
    
    if (!$result['success']) {
        sendError($result['error'] ?? 'Failed to delete thread', 500);
    }
    
    // Log success
    logRequest('thread_delete', $threadId);
    
    // Send response
    sendResponse([
        'success' => true,
        'thread_id' => $threadId,
        'message' => 'Thread and all associated data deleted successfully'
    ], 200);
    
} catch (Exception $e) {
    error_log('API Error (thread_delete): ' . $e->getMessage());
    sendError('Internal server error', 500);
}

