<?php
/**
 * API v1: Create Thread
 * 
 * POST /api/v1/thread_create.php
 * 
 * Body (JSON):
 * {
 *   "metadata": {
 *     "user_id": "12345",
 *     "app_name": "MyApp"
 *   }
 * }
 * 
 * Response:
 * {
 *   "success": true,
 *   "thread_id": "thread_abc123...",
 *   "private_key": "key_xyz789...",
 *   "created_at": "2025-12-18 14:30:00"
 * }
 */

require_once __DIR__ . '/../config.php';

use Redact\Classes\ThreadManager;

try {
    logRequest('thread_create');
    
    // Only allow POST
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        sendError('Method not allowed. Use POST', 405);
    }
    
    // Get request body
    $body = getRequestBody();
    $metadata = $body['metadata'] ?? [];
    
    // Add API metadata
    $metadata['api_version'] = API_VERSION;
    $metadata['created_via'] = 'api';
    $metadata['ip_address'] = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
    
    // Create thread
    $threadManager = new ThreadManager();
    $result = $threadManager->createThread($metadata);
    
    if (!$result['success']) {
        sendError('Failed to create thread', 500);
    }
    
    $threadId = $result['thread_id'];
    
    // Generate private key
    $privateKey = 'key_' . bin2hex(random_bytes(32));
    
    // Save authentication data
    $authFile = __DIR__ . '/../../data/' . $threadId . '/auth.json';
    $authData = [
        'thread_id' => $threadId,
        'private_key' => $privateKey,
        'created_at' => date('Y-m-d H:i:s'),
        'created_via' => 'api',
        'ip_address' => $metadata['ip_address']
    ];
    
    file_put_contents($authFile, json_encode($authData, JSON_PRETTY_PRINT));
    
    // Log success
    logRequest('thread_create', $threadId);
    
    // Send response
    sendResponse([
        'success' => true,
        'thread_id' => $threadId,
        'private_key' => $privateKey,
        'created_at' => $result['thread']['created_at'],
        'message' => 'Thread created successfully. Keep the private_key secure!'
    ], 201);
    
} catch (Exception $e) {
    error_log('API Error (thread_create): ' . $e->getMessage());
    sendError('Internal server error', 500, $e->getMessage());
}

