<?php
/**
 * Unit Tests for Sample Files
 * Tests real-world files to ensure proper processing
 */

require_once __DIR__ . '/../src/classes/autoload.php';

use Redact\Classes\FileHandler;

class SampleFilesTest
{
    private array $results = [];
    private string $samplesDir;
    
    public function __construct()
    {
        $this->samplesDir = __DIR__ . '/Samples';
    }
    
    public function run(): void
    {
        echo "SampleFilesTest:\n";
        
        $this->testSamplePDFExists();
        $this->testSamplePDFConversion();
        $this->testSamplePDFPageCount();
        $this->testSamplePDFPagesAreSeparate();
        $this->testSamplePDFImageQuality();
        $this->testSamplePDFMimeDetection();
        
        $this->displayResults();
    }
    
    private function testSamplePDFExists(): void
    {
        $sampleFile = $this->samplesDir . '/UPDATED Regulated Employment Reference.pdf';
        
        $this->assert(
            file_exists($sampleFile),
            'Sample PDF file exists in Unit/Samples/'
        );
        
        if (file_exists($sampleFile)) {
            $fileSize = filesize($sampleFile);
            $this->assert(
                $fileSize > 0,
                'Sample PDF has valid file size (' . number_format($fileSize) . ' bytes)'
            );
        }
    }
    
    private function testSamplePDFConversion(): void
    {
        // Skip if Imagick is not available
        if (!extension_loaded('imagick')) {
            $this->skip('Sample PDF converts successfully (requires Imagick)');
            return;
        }
        
        $sampleFile = $this->samplesDir . '/UPDATED Regulated Employment Reference.pdf';
        
        if (!file_exists($sampleFile)) {
            $this->skip('Sample PDF converts successfully (file not found)');
            return;
        }
        
        $handler = new FileHandler();
        $result = $handler->convertToImages($sampleFile, 'UPDATED Regulated Employment Reference.pdf');
        
        $this->assert(
            $result['success'] === true,
            'Sample PDF converts successfully without errors'
        );
        
        if (!$result['success']) {
            $this->assert(false, 'Conversion error details', $result['error'] ?? 'Unknown error');
        }
    }
    
    private function testSamplePDFPageCount(): void
    {
        // Skip if Imagick is not available
        if (!extension_loaded('imagick')) {
            $this->skip('Sample PDF reports correct page count (requires Imagick)');
            return;
        }
        
        $sampleFile = $this->samplesDir . '/UPDATED Regulated Employment Reference.pdf';
        
        if (!file_exists($sampleFile)) {
            $this->skip('Sample PDF reports correct page count (file not found)');
            return;
        }
        
        $handler = new FileHandler();
        $result = $handler->convertToImages($sampleFile, 'UPDATED Regulated Employment Reference.pdf');
        
        if (!$result['success']) {
            $this->skip('Sample PDF reports correct page count (conversion failed)');
            return;
        }
        
        $pageCount = $result['page_count'] ?? 0;
        $imageCount = count($result['images'] ?? []);
        
        $this->assert(
            $pageCount > 0,
            "Sample PDF has pages (detected: $pageCount pages)"
        );
        
        $this->assert(
            $imageCount === $pageCount,
            "Number of images matches page count ($imageCount images = $pageCount pages)"
        );
    }
    
    private function testSamplePDFPagesAreSeparate(): void
    {
        // Skip if Imagick is not available
        if (!extension_loaded('imagick')) {
            $this->skip('Sample PDF pages are converted separately (requires Imagick)');
            return;
        }
        
        $sampleFile = $this->samplesDir . '/UPDATED Regulated Employment Reference.pdf';
        
        if (!file_exists($sampleFile)) {
            $this->skip('Sample PDF pages are converted separately (file not found)');
            return;
        }
        
        $handler = new FileHandler();
        $result = $handler->convertToImages($sampleFile, 'UPDATED Regulated Employment Reference.pdf');
        
        if (!$result['success'] || !isset($result['images']) || count($result['images']) < 2) {
            $this->skip('Sample PDF pages are converted separately (not multi-page or conversion failed)');
            return;
        }
        
        $images = $result['images'];
        
        // Verify all images are different
        $allDifferent = true;
        for ($i = 0; $i < count($images) - 1; $i++) {
            if ($images[$i] === $images[$i + 1]) {
                $allDifferent = false;
                break;
            }
        }
        
        $this->assert(
            $allDifferent,
            'All pages produce DIFFERENT images (not duplicated)'
        );
        
        // Verify each image is single-page
        try {
            $allSinglePage = true;
            foreach ($images as $index => $imageBlob) {
                $img = new \Imagick();
                $img->readImageBlob($imageBlob);
                $numImages = $img->getNumberImages();
                $img->clear();
                
                if ($numImages !== 1) {
                    $allSinglePage = false;
                    break;
                }
            }
            
            $this->assert(
                $allSinglePage,
                'Each converted image is SINGLE-PAGE (flattened page-by-page, not combined)'
            );
        } catch (Exception $e) {
            $this->assert(false, 'Verify single-page images', $e->getMessage());
        }
    }
    
    private function testSamplePDFImageQuality(): void
    {
        // Skip if Imagick is not available
        if (!extension_loaded('imagick')) {
            $this->skip('Sample PDF produces high-quality images (requires Imagick)');
            return;
        }
        
        $sampleFile = $this->samplesDir . '/UPDATED Regulated Employment Reference.pdf';
        
        if (!file_exists($sampleFile)) {
            $this->skip('Sample PDF produces high-quality images (file not found)');
            return;
        }
        
        $handler = new FileHandler();
        $result = $handler->convertToImages($sampleFile, 'UPDATED Regulated Employment Reference.pdf');
        
        if (!$result['success'] || !isset($result['images']) || count($result['images']) === 0) {
            $this->skip('Sample PDF produces high-quality images (conversion failed)');
            return;
        }
        
        try {
            $firstImage = $result['images'][0];
            $img = new \Imagick();
            $img->readImageBlob($firstImage);
            
            $width = $img->getImageWidth();
            $height = $img->getImageHeight();
            $format = $img->getImageFormat();
            $quality = $img->getImageCompressionQuality();
            
            $img->clear();
            
            // Verify reasonable dimensions (should be letter size at 300 DPI)
            $this->assert(
                $width >= 2000 && $height >= 2000,
                "Images have sufficient resolution ({$width}x{$height} pixels at 300 DPI)"
            );
            
            $this->assert(
                strtolower($format) === 'jpeg',
                "Images are in JPEG format (actual: $format)"
            );
            
            $this->assert(
                $quality >= 90,
                "Images have high quality setting (quality: $quality/100)"
            );
            
        } catch (Exception $e) {
            $this->assert(false, 'Verify image quality', $e->getMessage());
        }
    }
    
    private function testSamplePDFMimeDetection(): void
    {
        $sampleFile = $this->samplesDir . '/UPDATED Regulated Employment Reference.pdf';
        
        if (!file_exists($sampleFile)) {
            $this->skip('Sample PDF MIME type detected correctly (file not found)');
            return;
        }
        
        // Skip if Imagick is not available
        if (!extension_loaded('imagick')) {
            $this->skip('Sample PDF MIME type detected correctly (requires Imagick)');
            return;
        }
        
        $handler = new FileHandler();
        $result = $handler->convertToImages($sampleFile, 'UPDATED Regulated Employment Reference.pdf');
        
        if (!$result['success']) {
            $this->skip('Sample PDF MIME type detected correctly (conversion failed)');
            return;
        }
        
        $this->assert(
            isset($result['detected_type']) && $result['detected_type'] === 'application/pdf',
            'Sample file correctly detected as PDF (MIME: application/pdf)'
        );
    }
    
    private function assert(bool $condition, string $message, string $error = ''): void
    {
        $this->results[] = [
            'passed' => $condition,
            'message' => $message,
            'error' => $error
        ];
        
        $status = $condition ? '✓' : '✗';
        $output = "  $status $message";
        if (!$condition && $error) {
            $output .= " - Error: $error";
        }
        echo "$output\n";
    }
    
    private function skip(string $message): void
    {
        $this->results[] = [
            'passed' => true,
            'message' => $message,
            'skipped' => true
        ];
        echo "  ⊘ $message (skipped)\n";
    }
    
    private function displayResults(): void
    {
        $passed = count(array_filter($this->results, fn($r) => $r['passed']));
        $total = count($this->results);
        echo "\n";
    }
}

// Run tests if executed directly
if (basename(__FILE__) === basename($_SERVER['PHP_SELF'])) {
    $test = new SampleFilesTest();
    $test->run();
}
