<?php
/**
 * Unit Tests for RegistryManager
 */

require_once __DIR__ . '/../src/classes/autoload.php';

use Redact\Classes\RegistryManager;

class RegistryManagerTest
{
    private array $results = [];
    private string $testDataDir;
    
    public function __construct()
    {
        $this->testDataDir = __DIR__ . '/test_data/registry';
        if (!is_dir($this->testDataDir)) {
            mkdir($this->testDataDir, 0755, true);
        }
    }
    
    public function run(): void
    {
        echo "RegistryManagerTest:\n";
        
        $this->testConstructor();
        $this->testConstructorWithThread();
        $this->testBuildRegistriesFromBlocks();
        $this->testGetLayoutsToProcess();
        $this->testMapPIIToWordBlocks();
        $this->testApplyPIIToPage();
        $this->testCacheFunctionality();
        $this->testGetStatistics();
        $this->testClearCache();
        
        $this->displayResults();
    }
    
    private function testConstructor(): void
    {
        try {
            $manager = new RegistryManager();
            $this->assert(true, 'Constructor without parameters');
        } catch (Exception $e) {
            $this->assert(false, 'Constructor without parameters', $e->getMessage());
        }
    }
    
    private function testConstructorWithThread(): void
    {
        try {
            $manager = new RegistryManager('test_thread_123', $this->testDataDir);
            $this->assert(true, 'Constructor with thread context');
        } catch (Exception $e) {
            $this->assert(false, 'Constructor with thread context', $e->getMessage());
        }
    }
    
    private function testBuildRegistriesFromBlocks(): void
    {
        $manager = new RegistryManager();
        
        // Mock Textract blocks
        $mockBlocks = [
            [
                'Id' => 'layout-1',
                'BlockType' => 'LAYOUT_TEXT',
                'Confidence' => 99.5,
                'Geometry' => [
                    'BoundingBox' => [
                        'Left' => 0.1,
                        'Top' => 0.1,
                        'Width' => 0.5,
                        'Height' => 0.1
                    ]
                ],
                'Relationships' => [
                    [
                        'Type' => 'CHILD',
                        'Ids' => ['word-1', 'word-2']
                    ]
                ]
            ],
            [
                'Id' => 'word-1',
                'BlockType' => 'WORD',
                'Text' => 'Hello',
                'Confidence' => 99.0,
                'Geometry' => [
                    'BoundingBox' => [
                        'Left' => 0.1,
                        'Top' => 0.1,
                        'Width' => 0.1,
                        'Height' => 0.05
                    ]
                ]
            ],
            [
                'Id' => 'word-2',
                'BlockType' => 'WORD',
                'Text' => 'World',
                'Confidence' => 98.5,
                'Geometry' => [
                    'BoundingBox' => [
                        'Left' => 0.2,
                        'Top' => 0.1,
                        'Width' => 0.1,
                        'Height' => 0.05
                    ]
                ]
            ]
        ];
        
        [$layouts, $words] = $manager->buildRegistriesFromBlocks($mockBlocks, 1);
        
        $this->assert(
            is_array($layouts) && count($layouts) === 1,
            'buildRegistriesFromBlocks extracts layouts'
        );
        
        $this->assert(
            is_array($words) && count($words) === 2,
            'buildRegistriesFromBlocks extracts word blocks'
        );
    }
    
    private function testGetLayoutsToProcess(): void
    {
        $manager = new RegistryManager();
        
        $mockBlocks = [
            [
                'Id' => 'layout-1',
                'BlockType' => 'LAYOUT_TEXT',
                'Confidence' => 99.5,
                'Geometry' => [
                    'BoundingBox' => [
                        'Left' => 0.1,
                        'Top' => 0.1,
                        'Width' => 0.5,
                        'Height' => 0.1
                    ]
                ],
                'Relationships' => [
                    [
                        'Type' => 'CHILD',
                        'Ids' => ['word-1']
                    ]
                ]
            ],
            [
                'Id' => 'word-1',
                'BlockType' => 'WORD',
                'Text' => 'Test',
                'Confidence' => 99.0,
                'Geometry' => [
                    'BoundingBox' => [
                        'Left' => 0.1,
                        'Top' => 0.1,
                        'Width' => 0.1,
                        'Height' => 0.05
                    ]
                ]
            ]
        ];
        
        $manager->buildRegistriesFromBlocks($mockBlocks, 1);
        $layoutsToProcess = $manager->getLayoutsToProcess();
        
        $this->assert(
            is_array($layoutsToProcess) && count($layoutsToProcess) > 0,
            'getLayoutsToProcess identifies unprocessed layouts'
        );
    }
    
    private function testMapPIIToWordBlocks(): void
    {
        $manager = new RegistryManager();
        
        // Build registry first
        $mockBlocks = [
            [
                'Id' => 'layout-1',
                'BlockType' => 'LAYOUT_TEXT',
                'Confidence' => 99.5,
                'Geometry' => [
                    'BoundingBox' => [
                        'Left' => 0.1,
                        'Top' => 0.1,
                        'Width' => 0.5,
                        'Height' => 0.1
                    ]
                ],
                'Relationships' => [
                    [
                        'Type' => 'CHILD',
                        'Ids' => ['word-1']
                    ]
                ]
            ],
            [
                'Id' => 'word-1',
                'BlockType' => 'WORD',
                'Text' => 'John',
                'Confidence' => 99.0,
                'Geometry' => [
                    'BoundingBox' => [
                        'Left' => 0.1,
                        'Top' => 0.1,
                        'Width' => 0.1,
                        'Height' => 0.05
                    ]
                ]
            ]
        ];
        
        $manager->buildRegistriesFromBlocks($mockBlocks, 1);
        
        // Mock PII entities
        $mockEntities = [
            [
                'Type' => 'NAME',
                'Score' => 0.99,
                'BeginOffset' => 0,
                'EndOffset' => 4
            ]
        ];
        
        $layoutText = 'John';
        $manager->mapPIIToWordBlocks($mockEntities, 'page1_layout0', $layoutText);
        
        $this->assert(true, 'mapPIIToWordBlocks maps PII to word blocks');
    }
    
    private function testApplyPIIToPage(): void
    {
        $manager = new RegistryManager();
        
        // Build registry with PII
        $mockBlocks = [
            [
                'Id' => 'layout-1',
                'BlockType' => 'LAYOUT_TEXT',
                'Confidence' => 99.5,
                'Geometry' => [
                    'BoundingBox' => [
                        'Left' => 0.1,
                        'Top' => 0.1,
                        'Width' => 0.5,
                        'Height' => 0.1
                    ]
                ],
                'Relationships' => [
                    [
                        'Type' => 'CHILD',
                        'Ids' => ['word-1']
                    ]
                ]
            ],
            [
                'Id' => 'word-1',
                'BlockType' => 'WORD',
                'Text' => 'John',
                'Confidence' => 99.0,
                'Geometry' => [
                    'BoundingBox' => [
                        'Left' => 0.1,
                        'Top' => 0.1,
                        'Width' => 0.1,
                        'Height' => 0.05
                    ]
                ]
            ]
        ];
        
        [$layouts, $words] = $manager->buildRegistriesFromBlocks($mockBlocks, 1);
        
        // Map PII
        $mockEntities = [
            [
                'Type' => 'NAME',
                'Score' => 0.99,
                'BeginOffset' => 0,
                'EndOffset' => 4
            ]
        ];
        $manager->mapPIIToWordBlocks($mockEntities, 'page1_layout0', 'John');
        
        // Apply to page
        $pageData = [
            'page_number' => 1,
            'word_blocks' => $words
        ];
        
        $piiBlocks = $manager->applyPIIToPage($pageData);
        
        $this->assert(
            is_array($piiBlocks) && count($piiBlocks) > 0,
            'applyPIIToPage returns PII blocks for page'
        );
    }
    
    private function testCacheFunctionality(): void
    {
        $manager = new RegistryManager('test_thread_cache', $this->testDataDir);
        
        // Build some data
        $mockBlocks = [
            [
                'Id' => 'word-1',
                'BlockType' => 'WORD',
                'Text' => 'Test',
                'Confidence' => 99.0,
                'Geometry' => [
                    'BoundingBox' => [
                        'Left' => 0.1,
                        'Top' => 0.1,
                        'Width' => 0.1,
                        'Height' => 0.05
                    ]
                ]
            ]
        ];
        
        $manager->buildRegistriesFromBlocks($mockBlocks, 1);
        
        // Save cache
        $saved = $manager->saveCachedRegistry();
        $this->assert($saved === true, 'saveCachedRegistry saves data to cache');
        
        // Load cache
        $manager2 = new RegistryManager('test_thread_cache', $this->testDataDir);
        $stats = $manager2->getCacheStatistics();
        
        $this->assert(
            $stats['cache_exists'] === true,
            'loadCachedRegistry loads cached data'
        );
    }
    
    private function testGetStatistics(): void
    {
        $manager = new RegistryManager();
        
        $stats = $manager->getStatistics();
        
        $this->assert(
            is_array($stats) && isset($stats['total_layouts']) && isset($stats['unique_words']),
            'getStatistics returns comprehensive statistics'
        );
    }
    
    private function testClearCache(): void
    {
        $manager = new RegistryManager('test_thread_clear', $this->testDataDir);
        
        // Save something
        $manager->saveCachedRegistry();
        
        // Clear cache
        $cleared = $manager->clearCache();
        
        $this->assert($cleared === true, 'clearCache removes cached data');
        
        $stats = $manager->getCacheStatistics();
        $this->assert(
            $stats['cached_words'] === 0,
            'Cache is empty after clearing'
        );
    }
    
    private function assert(bool $condition, string $message, string $error = ''): void
    {
        $this->results[] = [
            'passed' => $condition,
            'message' => $message,
            'error' => $error
        ];
        
        $status = $condition ? '✓' : '✗';
        $output = "  $status $message";
        if (!$condition && $error) {
            $output .= " - Error: $error";
        }
        echo "$output\n";
    }
    
    private function displayResults(): void
    {
        $passed = count(array_filter($this->results, fn($r) => $r['passed']));
        $total = count($this->results);
        echo "\n";
    }
    
    public function __destruct()
    {
        // Cleanup test data
        if (is_dir($this->testDataDir)) {
            $files = glob($this->testDataDir . '/*');
            foreach ($files as $file) {
                if (is_file($file)) {
                    unlink($file);
                }
            }
        }
    }
}

// Run tests if executed directly
if (basename(__FILE__) === basename($_SERVER['PHP_SELF'])) {
    $test = new RegistryManagerTest();
    $test->run();
}
