<?php
/**
 * Unit Tests for ComprehendService
 */

require_once __DIR__ . '/../src/classes/autoload.php';

use Redact\Classes\ComprehendService;

class ComprehendServiceTest
{
    private ?ComprehendService $service = null;
    private array $results = [];
    
    public function __construct()
    {
        // Load config if available (for optional integration tests)
        $configFile = __DIR__ . '/../src/config/config.php';
        if (file_exists($configFile)) {
            require_once $configFile;
        }
    }
    
    public function run(): void
    {
        echo "ComprehendServiceTest:\n";
        
        $this->testConstructor();
        $this->testConstructorWithRegion();
        $this->testDetectPiiEntitiesStructure();
        $this->testEmptyText();
        $this->testLongText();
        $this->testNonEnglishLanguage();
        
        $this->displayResults();
    }
    
    private function testConstructor(): void
    {
        try {
            $service = new ComprehendService('test_key', 'test_secret');
            $this->assert(true, 'Constructor with default region');
        } catch (Exception $e) {
            $this->assert(false, 'Constructor with default region', $e->getMessage());
        }
    }
    
    private function testConstructorWithRegion(): void
    {
        try {
            $service = new ComprehendService('test_key', 'test_secret', 'us-west-2');
            $this->assert(true, 'Constructor with custom region');
        } catch (Exception $e) {
            $this->assert(false, 'Constructor with custom region', $e->getMessage());
        }
    }
    
    private function testDetectPiiEntitiesStructure(): void
    {
        $service = new ComprehendService('test_key', 'test_secret');
        $result = $service->detectPiiEntities('Test text with John Doe');
        
        // Should return array with success key
        $this->assert(
            is_array($result) && isset($result['success']),
            'detectPiiEntities returns structured response'
        );
        
        // With invalid credentials, should return success=false
        $this->assert(
            $result['success'] === false,
            'detectPiiEntities returns false with invalid credentials'
        );
        
        // Should have error information
        $this->assert(
            isset($result['error']) || isset($result['curl_error']),
            'detectPiiEntities includes error details on failure'
        );
    }
    
    private function testEmptyText(): void
    {
        $service = new ComprehendService('test_key', 'test_secret');
        $result = $service->detectPiiEntities('');
        
        $this->assert(
            is_array($result),
            'detectPiiEntities handles empty text'
        );
    }
    
    private function testLongText(): void
    {
        $service = new ComprehendService('test_key', 'test_secret');
        $longText = str_repeat('This is a test text. ', 500); // ~10KB
        $result = $service->detectPiiEntities($longText);
        
        $this->assert(
            is_array($result) && isset($result['success']),
            'detectPiiEntities handles long text (5000 words)'
        );
    }
    
    private function testNonEnglishLanguage(): void
    {
        $service = new ComprehendService('test_key', 'test_secret');
        $result = $service->detectPiiEntities('Bonjour monde', 'fr');
        
        $this->assert(
            is_array($result) && isset($result['success']),
            'detectPiiEntities accepts non-English language code'
        );
    }
    
    // Integration test (requires valid AWS credentials)
    private function testRealAPICall(): void
    {
        // Skip if no credentials
        if (!defined('AWS_KEY') || !defined('AWS_SECRET')) {
            $this->skip('Real API call (requires AWS credentials)');
            return;
        }
        
        $service = new ComprehendService(AWS_KEY, AWS_SECRET, AWS_REGION);
        $result = $service->detectPiiEntities('My name is John Doe and my email is john@example.com');
        
        if ($result['success']) {
            $this->assert(
                isset($result['data']['Entities']),
                'Real API call returns entities'
            );
        } else {
            $this->skip('Real API call (AWS returned error)');
        }
    }
    
    private function assert(bool $condition, string $message, string $error = ''): void
    {
        $this->results[] = [
            'passed' => $condition,
            'message' => $message,
            'error' => $error
        ];
        
        $status = $condition ? '✓' : '✗';
        $output = "  $status $message";
        if (!$condition && $error) {
            $output .= " - Error: $error";
        }
        echo "$output\n";
    }
    
    private function skip(string $message): void
    {
        echo "  ⊘ $message (skipped)\n";
    }
    
    private function displayResults(): void
    {
        $passed = count(array_filter($this->results, fn($r) => $r['passed']));
        $total = count($this->results);
        echo "\n";
    }
}

// Run tests if executed directly
if (basename(__FILE__) === basename($_SERVER['PHP_SELF'])) {
    $test = new ComprehendServiceTest();
    $test->run();
}
