<?php
/**
 * API v1: Process File
 * 
 * POST /api/v1/process_file.php
 * 
 * Body (JSON):
 * {
 *   "thread_id": "thread_abc123...",
 *   "private_key": "key_xyz789...",
 *   "file_data": "base64_encoded_file_content",
 *   "file_name": "document.pdf"
 * }
 * 
 * Response:
 * {
 *   "success": true,
 *   "thread_id": "thread_abc123...",
 *   "processing_time": "12345ms",
 *   "total_pages": 3,
 *   "total_pii_instances": 156,
 *   "comprehend_calls": 45,
 *   "optimization_rate": 65.2,
 *   "cache": {...},
 *   "pages": [...]
 * }
 */

require_once __DIR__ . '/../config.php';

use Redact\Classes\PIIDetectionService;

// Increase execution time and memory for file processing
ini_set('max_execution_time', 600);
ini_set('memory_limit', '512M');
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// Disable output buffering
if (ob_get_level()) {
    ob_end_clean();
}

error_log('process_file.php called at ' . date('Y-m-d H:i:s'));

try {
    logRequest('process_file');
    
    // Only allow POST
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        sendError('Method not allowed. Use POST', 405);
    }
    
    // Get request body
    $body = getRequestBody();
    $threadId = $body['thread_id'] ?? null;
    $privateKey = $body['private_key'] ?? null;
    $fileData = $body['file_data'] ?? null;
    $fileName = $body['file_name'] ?? 'document.pdf';
    
    // Validate required fields
    if (empty($fileData)) {
        sendError('file_data is required (base64 encoded)', 400);
    }
    
    if (empty($fileName)) {
        sendError('file_name is required', 400);
    }
    
    // Validate authentication
    validateThreadAuth($threadId, $privateKey);
    
    // Decode base64 file data
    $decodedData = base64_decode($fileData, true);
    
    if ($decodedData === false) {
        sendError('Invalid base64 encoding in file_data', 400);
    }
    
    // Check file size
    $fileSize = strlen($decodedData);
    if ($fileSize > API_MAX_FILE_SIZE) {
        $maxMB = API_MAX_FILE_SIZE / 1024 / 1024;
        sendError("File size exceeds {$maxMB}MB limit", 400);
    }
    
    if ($fileSize === 0) {
        sendError('File data is empty', 400);
    }
    
    // Save to temporary file
    $tempFile = sys_get_temp_dir() . '/' . uniqid('pii_api_', true) . '_' . basename($fileName);
    file_put_contents($tempFile, $decodedData);
    
    // Initialize PII Detection Service
    $awsCredentials = getConfig('AWS_Credentials');
    $piiService = new PIIDetectionService($awsCredentials, [
        'region' => 'us-east-1',
        'max_file_size' => API_MAX_FILE_SIZE
    ]);
    
    // Process document
    $result = $piiService->processDocument($tempFile, $threadId);
    
    // Clean up temporary file
    if (file_exists($tempFile)) {
        unlink($tempFile);
    }
    
    // Check result
    if (!$result['success']) {
        sendError($result['error'] ?? 'File processing failed', 500);
    }
    
    // Log success
    logRequest('process_file', $threadId);
    
    // Debug: Log result size before sending
    error_log('About to send response. Result keys: ' . implode(', ', array_keys($result)));
    error_log('Total pages in result: ' . ($result['total_pages'] ?? 'not set'));
    
    // Ensure result is not empty
    if (empty($result)) {
        error_log('ERROR: Result is empty!');
        sendError('Processing completed but result is empty', 500);
    }
    
    // Send response
    sendResponse($result, 200);
    
} catch (Exception $e) {
    error_log('API Error (process_file): ' . $e->getMessage());
    error_log('Stack trace: ' . $e->getTraceAsString());
    
    // Clean up temp file on error
    if (isset($tempFile) && file_exists($tempFile)) {
        unlink($tempFile);
    }
    
    sendError('Internal server error: ' . $e->getMessage(), 500);
}

